import { requireAdmin, adminError } from "@/lib/admin";
import { listInvites } from "@/lib/store";
import { inviteUrl } from "@/lib/links";
import { sendInviteEmail, emailConfigured } from "@/lib/notify/email";
import { sendSms, sendWhatsapp, twilioConfigured } from "@/lib/notify/twilio";
import { saveRsvp } from "@/lib/store"; // not used but keep store import pattern

export async function POST(req, { params }) {
  if (!requireAdmin(req)) return adminError();

  const body = await req.json();
  const channels = body.channels || { email: true, sms: false, whatsapp: false };
  const subject = body.subject || "Ftesa per dasme";
  const message = body.message || "Ju lutem hapni ftesen ne linkun me poshte:";
  const limit = Math.min(parseInt(body.limit || "50", 10) || 50, 500);

  const invites = (await listInvites(params.eventId)).slice(0, limit);

  const results = [];
  const hasEmail = emailConfigured();
  const hasTwilio = twilioConfigured();

  for (const inv of invites) {
    const link = inviteUrl(inv.token);
    const guestName = inv.guest?.name || "Mik";
    const bodyText = `${message}\n\n${link}\n\nMe respekt,\nFtesa Digjitale`;

    const row = { token: inv.token, guest: inv.guest, link, sent: {} };

    if (channels.email && inv.guest?.email) {
      if (!hasEmail) {
        row.sent.email = { ok: false, reason: "smtp_not_configured" };
      } else {
        row.sent.email = await sendInviteEmail({
          to: inv.guest.email,
          subject,
          text: bodyText,
          html: `<p>Pershendetje ${escapeHtml(guestName)},</p><p>${escapeHtml(message)}</p><p><a href="${link}">${link}</a></p><p>Me respekt,<br/>Ftesa Digjitale</p>`,
        });
      }
    }

    if (channels.sms && inv.guest?.phone) {
      if (!hasTwilio) {
        row.sent.sms = { ok: false, reason: "twilio_not_configured" };
      } else {
        row.sent.sms = await sendSms({ to: inv.guest.phone, body: bodyText });
      }
    }

    if (channels.whatsapp && inv.guest?.phone) {
      if (!hasTwilio) {
        row.sent.whatsapp = { ok: false, reason: "twilio_not_configured" };
      } else {
        row.sent.whatsapp = await sendWhatsapp({ to: inv.guest.phone, body: bodyText });
      }
    }

    results.push(row);
  }

  return new Response(JSON.stringify({
    ok: true,
    configured: { email: hasEmail, twilio: hasTwilio },
    processed: results.length,
    results,
  }), { headers: { "content-type": "application/json" } });
}

function escapeHtml(s) {
  return String(s).replace(/[&<>"']/g, (c) => ({
    "&": "&amp;",
    "<": "&lt;",
    ">": "&gt;",
    '"': "&quot;",
    "'": "&#039;",
  }[c]));
}
